
import java.io.*;
import java.util.*;


/****************************************************************
 *Class for Java Console input, with methods for reading one
 *input value per line. If the user enters an improper input,
 *i.e., an input of the wrong type or a blank line, they is
 *prompted to reenter the input and is given a brief explanation
 *of what is required. Also includes some additional methods to
 *input single numbers, words, and characters, without going to
 *the next line.
 */
public class JLiveRead {

/** Read (the rest of) a line of text and return it as a String. 
    The ending '\n' or "\r\n" is not included in the value. It is
    assumed that '\r' is always directly followed by '\n'
    */
public static String readString() { 
    char nextChar= readChar();
    String res= "";
    boolean present= false;
    
    /*invariant: the String read thus far (with any '\r' character 
       removed) is res + nextChar.
       */
    while (nextChar != '\n') {
        if (nextChar != '\r')
            res= res + nextChar;
     	nextChar= readChar();
        }
    return res;
    }

/** Read and return the first sequence of nonwhite characters on a 
    line. The rest of the line is discarded. If the line contains only 
    white space, the user is asked to reenter the line.
    */
public static String readLineString() {
    String s= readString();
    StringTokenizer st= new StringTokenizer(s);
    
    /* invariant: s contains the last line read and st contains its
       tokenized form. All previously read lines has only whitespace
       */
    while(!st.hasMoreTokens()) {
        System.out.println(
           "Input line must contain at least one non-whitespace");
        System.out.println(
           "character. Try again: Enter input:");
        s= readString();
        st= new StringTokenizer(s);
        }
    return st.nextToken();
    }
    
/** The input line is supposed to contain a single int integer,
    perhaps with whitespace on either side. Read the line and
    return the integer on it. If the line does not contain an
    int integer, ask the user to try again.
    */
public static int readLineInt() {
    String input= readString().trim();
    // invariant: input contains the last input line read; previous
    // lines did not contain an recognizable integer.
    while (true) {
        try {return Integer.valueOf(input).intValue();
           }
        catch (NumberFormatException e) {
           System.out.println(
               "Input is not an int. It must be an int like");
           System.out.println("43 or -20. Try again: enter an integer:");
           input= readString().trim();
           }
        } 
    }
  
/** The input line is supposed to contain a single long integer,
    perhaps with whitespace on either side. Read the line and
    return the integer on it. If the line does not contain a
    long integer, ask the user to try again.
    */
public static long readLineLong() {
    String input= readString().trim();
    // invariant: input contains the last input line read; previous
    // lines did not contain an recognizable integer.
    while (true) {
        try {return Long.valueOf(input).longValue();
           }
        catch (NumberFormatException e) {
           System.out.println(
               "Input is not a long. It must be a long integer like");
           System.out.println("43 or -20. Try again: enter an integer:");
           input= readString().trim();
           }
        } 
    }
    
/** The input line is supposed to contain a single double value, perhaps 
    with whitespace on either side. Read the line and return the double.
    If the line  does not contain a double, ask the user to try again.
    */
public static double readLineDouble() {
    String input= readString().trim();
    // invariant: input contains the last input line read; previous
    // lines did not contain an recognizable integer.
    while (true) {
        try {return Double.valueOf(input).doubleValue();
           }
        catch (NumberFormatException e) {
           System.out.println(
               "Input is not a double. It must be an number with or without a");
           System.out.println("decimal point, like 43.3. Try again: enter a double:");
           input= readString().trim();
           }
        } 
    }
     
/** The input line is supposed to contain a single float value, perhaps
    with whitespace on either side. Read the line and return the float.
    If the line does not contain a float, ask the user to try again.
    */
public static float readLineFloat() {
    String input= readString().trim();
    // invariant: input contains the last input line read; previous
    // lines did not contain an recognizable integer.
    while (true) {
        try {return Float.valueOf(input).floatValue();
           }
        catch (NumberFormatException e) {
           System.out.println(
               "Input is not a float. It must be an number with or without a");
           System.out.println("decimal point, like 43.3. Try again: enter a float:");
           input= readString().trim();
           }
        } 
    }

/** Read and return the first non-white character on a line and discard the
    rest of the line. If the line contains only whitespace, ask the user to
    type the line again.
    */
public static char readLineNonwhiteChar() {
    String input= readString().trim();
    while (input.length() == 0) {
        System.out.println("The input line did not contain non-whitespace");
        System.out.println("Try again: Enter input:");
        input= readString().trim();
       	}
    return input.charAt(0);
    }

/** Input should contain a single word: t or true or f or false (in
    upper or lower case). Return true or false, accordingly. If the
    line contains anything else, ask the user to reenter input.
    */ 
public static boolean readLineBoolean() {
    String input= readString().trim();
    //invariant: input contains the last input line with whitespace
    //   on either side removed. Previously read lines did not 
    // contain the necessary input.
    while ( !input.equalsIgnoreCase("true") &&
            !input.equalsIgnoreCase("t") &&
            !input.equalsIgnoreCase("false") &&
            !input.equalsIgnoreCase("f") ){
       System.out.println("Input was not one of following: true, t, false, f");
       System.out.println("(in uppercase or lower case).");
       System.out.println("Try again: Enter input:");
       input= readString().trim();
       }
        
    if (input.equalsIgnoreCase("true")) return true;
    if (input.equalsIgnoreCase("t")) return true;
    return false;
    }
 

/** Read and return the next input character (only). If there is
    an IO error, give message and terminate program.
    */
public static char readChar() {
    try {return (char) System.in.read();
       }
    catch(IOException e) {
       System.out.println(e.getMessage());
       System.out.println("Fatal error. Ending Program.");
       System.exit(0);          
       } 
        
    return ' ';         
    }
 
/** Read and return the next non-whitespace input character. If there 
    is anIO error, give message and terminate program.
    */
public static char readNonwhiteChar()  { 
    char c= readChar();
 
    // invariant: c is the last character read and all previous ones
    //            were whitespace 
    while (Character.isWhitespace(c))
        {c=  readChar();}
      
    return c;
    }

/** The next input begins with an int value, possibly preceded by
    whitespace and definitely followed by whitespace. Read and return
    the int value and discard the following whitespace character. The
    discarded characater may be '\r' or '\n'. If the next word doesn't
    represent an int value, throw a NumberFormatException.
    */
public static int readInt() throws NumberFormatException { 
    String input=  readWord(); 
    return (Integer.valueOf(input).intValue()); 
    }
   
/** The next input begins with a long value, possibly preceded by
    whitespace and definitely followed by whitespace. Read and return
    the long value and discard the following whitespace character. The
    discarded characater may be '\r' or '\n'. If the next word doesn't
    represent a long value, throw a NumberFormatException.
    */
public static long readLong() throws NumberFormatException { 
    String input=  readWord(); 
    return (Long.valueOf(input).longValue()); 
    }
    
/** The next input begins with a double value, possibly preceded by
    whitespace and definitely followed by whitespace. Read and return
    the double value and discard the following whitespace character. The
    discarded characater may be '\r' or '\n'. If the next word doesn't
    represent a double value, throw a NumberFormatException.
    */
public static double readDouble() throws NumberFormatException { 
    String input=  readWord(); 
    return (Double.valueOf(input).doubleValue()); 
    }
      
/** The next input begins with a float value, possibly preceded by
    whitespace and definitely followed by whitespace. Read and return
    the double value and discard the following whitespace character. The
    discarded characater may be '\r' or '\n'. If the next word doesn't
    represent a float value, throw a NumberFormatException.
    */
public static float readFloat() throws NumberFormatException { 
    String input=  readWord(); 
    return (Float.valueOf(input).floatValue()); 
    }

/** Read and return the first sequence of nonwhite characters (skipping
      over preceding whitespace). Discard the whitespace character 
      (which may be '\r\n' or '\n') that follows the String. Thus, empty
      lines are skipped. Whitespace char '\r' must be followed directly
      by '\n', and both are treated as a single character. Print a message
      and terminate program if '\r' is not followed by '\n'.
      */
public static String readWord() { 
    char c= readChar();
    //invariant: c is last char read and previous chars were whitespace    
    while (Character.isWhitespace(c))
        {c=  readChar();}
        
    String res= "" + c;
    c= readChar();
    //invariant: c contains last char read and res contains the
    //           nonwhitespace chars that preceded char.   
    while (!(Character.isWhitespace(c))) { 
        res= res + c; 
        c= readChar();
        }
    
    // res contains the String to be returned and c contains following char
        if (c == '\r') {
            c = readChar();
            if (c != '\n'){
                System.out.println("Fatal error in method JLiveRead.readString.");
                System.exit(1);
            }
        }

    return res;
    }

/* Read and return (as an int) the first byte in the input stream.
   Print message and terminate program if an IO exception occurs.
   Same as System.in.read(), except that it catches IOExceptions.
   */
public static int read() {
    try {return System.in.read();
        }
    catch(IOException e) {
        System.out.println(e.getMessage());
        System.out.println("Fatal error in method JLiveRead.read.");
        System.exit(0);          
        }
    return 0;
    }    
}

